<?php

/**
 * Gallery Widget for Elementor
 */
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}
class Gallery extends \Elementor\Widget_Base
{

    public function get_name()
    {
        return 'gallery_section';
    }

    public function get_title()
    {
        return esc_html__('Project Gallery', 'plugin-name');
    }

    public function get_icon()
    {
        return 'eicon-gallery-grid';
    }

    public function get_categories()
    {
        return ['dhaara'];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'section_header',
            [
                'label' => esc_html__('Header', 'plugin-name'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'background_text',
            [
                'label' => 'Background Text',
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'Gallery',
            ]
        );

        $this->add_control(
            'heading_text',
            [
                'label' => esc_html__('Heading', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'Project',
            ]
        );

        $this->add_control(
            'heading_span',
            [
                'label' => esc_html__('Heading Span Text', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'Gallery',
            ]
        );

        $this->add_control(
            'section_description',
            [
                'label' => esc_html__('Description', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => 'Explore our collection of stunning architectural photographs showcasing the beauty and innovation of our projects.',
            ]
        );

        $this->end_controls_section();

        // Filter Tabs
        $this->start_controls_section(
            'section_filters',
            [
                'label' => esc_html__('Filter Tabs', 'plugin-name'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'filter_name',
            [
                'label' => esc_html__('Filter Name', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'All Photos',
            ]
        );

        $repeater->add_control(
            'category',
            [
                'label' => esc_html__('Category Value', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'all',
                'description' => 'Value used for filtering (e.g., "exterior", "interior")'
            ]
        );

        $repeater->add_control(
            'category_color',
            [
                'label' => esc_html__('Category Color', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#60A5FA',
            ]
        );


        $repeater->add_control(
            'is_active',
            [
                'label' => esc_html__('Active', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'plugin-name'),
                'label_off' => esc_html__('No', 'plugin-name'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'filter_tabs',
            [
                'label' => esc_html__('Filter Tabs', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'filter_name' => 'All Photos',
                        'category' => 'all',
                        'category_color' => '#60A5FA',
                        'is_active' => 'yes'
                    ],
                    [
                        'filter_name' => 'Exterior',
                        'category' => 'exterior',
                        'category_color' => '#34D399'
                    ],
                    [
                        'filter_name' => 'Interior',
                        'category' => 'interior',
                        'category_color' => '#F472B6'
                    ],
                    [
                        'filter_name' => 'Amenities',
                        'category' => 'amenities',
                        'category_color' => '#A78BFA'
                    ],
                    [
                        'filter_name' => 'Construction',
                        'category' => 'construction',
                        'category_color' => '#FBBF24'
                    ],
                ],
                'title_field' => '{{{ filter_name }}}',
            ]
        );


        $this->end_controls_section();

        // Gallery Items
        $this->start_controls_section(
            'section_gallery',
            [
                'label' => esc_html__('Gallery Items', 'plugin-name'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $gallery_repeater = new \Elementor\Repeater();

        $gallery_repeater->add_control(
            'image',
            [
                'label' => esc_html__('Image', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $gallery_repeater->add_control(
            'category',
            [
                'label' => esc_html__('Category', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'description' => 'Enter category matching filter tabs (e.g., "exterior", "interior")',
                'default' => 'all',
            ]
        );

        $gallery_repeater->add_control(
            'category_color',
            [
                'label' => esc_html__('Category Color', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#60A5FA',
            ]
        );

        $gallery_repeater->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'Main Tower Facade',
            ]
        );

        $gallery_repeater->add_control(
            'description',
            [
                'label' => esc_html__('Description', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => 'Stunning glass facade with ocean views and modern architectural elements.',
            ]
        );

        $this->add_control(
            'gallery_items',
            [
                'label' => esc_html__('Gallery Items', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $gallery_repeater->get_controls(),
                'default' => [
                    [
                        'category' => 'exterior',
                        'title' => 'Main Tower Facade',
                        'description' => 'Stunning glass facade with ocean views and modern architectural elements.',
                    ],
                ],
                'title_field' => '{{{ title }}}',
            ]
        );


        $this->add_control(
            'load_more_text',
            [
                'label' => esc_html__('Load More Text', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'Load More Photos',
                'condition' => [
                    'show_load_more' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section
        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('Colors', 'plugin-name'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'primary_color',
            [
                'label' => esc_html__('Primary Color', 'plugin-name'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#FFB600',
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $primary_color = !empty($settings['primary_color']) ? $settings['primary_color'] : '#FFB600';
?>
        <section class="gallery-section relative py-16 sm:py-20 lg:py-20 overflow-hidden">
            <!-- Background Elements -->
            <div class="absolute inset-0 -z-10">
                <!-- Gradient Orbs -->
                <div class="absolute top-0 left-0 w-full h-full overflow-hidden">
                    <div style="background-color: <?php echo esc_attr($primary_color); ?>20" class="bg-fade-effect absolute top-1/4 -left-20 w-96 h-96 rounded-full mix-blend-multiply filter blur-3xl"></div>
                    <div style="background-color: <?php echo esc_attr($primary_color); ?>20" class="absolute bottom-1/4 right-0 w-96 h-96 rounded-full mix-blend-multiply filter blur-3xl animation-delay-2000"></div>
                </div>
            </div>

            <div class="max-w-7xl 2xl:max-w-8xl mx-auto px-4">
                <!-- Section Header -->
                <!-- Section Header -->
                <div class="relative mb-12 sm:mb-16 text-center z-10">
                    <div class="absolute -top-10 left-1/2  -translate-x-1/2  text-[80px] sm:text-[120px] font-bold text-gray-100 select-none opacity-0 md:opacity-60 leading-none -z-10">
                        <?php echo esc_html($settings['background_text']); ?>
                    </div>
                    <h2 class="relative text-3xl sm:text-4xl lg:text-5xl font-bold text-gray-900 fade-in-effect ">
                        <?php echo esc_html($settings['heading_text']); ?>
                        <span style="color: <?php echo esc_attr($primary_color); ?>"><?php echo esc_html($settings['heading_span']); ?></span>
                    </h2>
                    <p class="mt-4 text-xl text-gray-600 max-w-2xl mx-auto fade-in-effect effect-1">
                        <?php echo esc_html($settings['section_description']); ?>
                    </p>
                </div>
                <!-- Filter Tabs -->
                <div class="filter-buttons flex flex-wrap justify-center gap-6 sm:gap-8 mb-12 fade-in-effect effect-1" id="filterButtons">
                    <?php foreach ($settings['filter_tabs'] as $tab) : ?>
                        <button style="--active-color: <?php echo esc_attr($primary_color); ?>;" data-category="<?php echo esc_attr($tab['category']); ?>" class="filter-btn px-6 py-3 rounded-xl bg-white border border-gray-200 text-gray-600 font-medium hover:border-primary-amber transition-colors <?php echo $tab['is_active'] === 'yes' ? 'active-tab' : ''; ?>">
                            <?php echo esc_html($tab['filter_name']); ?>
                        </button>
                    <?php endforeach; ?>
                </div>


                <!-- Gallery Grid -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 sm:gap-8 fade-in-effect effect-2" id="galleryGrid">
                    <?php foreach ($settings['gallery_items'] as $item) :
                        // Find the matching filter tab to get the category color
                        $category_color = '#60A5FA'; // Default color
                        foreach ($settings['filter_tabs'] as $tab) {
                            if ($tab['category'] === $item['category']) {
                                $category_color = $tab['category_color'];
                                break;
                            }
                        }
                    ?>
                        <div class="group relative" data-category="<?php echo esc_attr($item['category']); ?>">
                            <div class="relative bg-white/80 backdrop-blur-xl rounded-2xl overflow-hidden shadow-lg hover:shadow-xl transition-all duration-500">
                                <!-- Image -->
                                <div class="relative aspect-[4/3] overflow-hidden">
                                    <?php
                                    if (!empty($item['image']['url'])) {
                                        echo wp_get_attachment_image($item['image']['id'], 'full', false, [
                                            'class' => 'w-full h-full object-cover transform group-hover:scale-110 transition-transform duration-700',
                                            'fetchpriority' => 'high',
                                            'decoding' => 'async'
                                        ]);
                                    }
                                    ?>

                                    <!-- Permanent gradient for title only -->
                                    <div class="absolute inset-x-0 bottom-0 h-40 bg-gradient-to-t from-black/90 to-transparent"></div>

                                    <!-- Hover gradient -->
                                    <div class="absolute inset-0 bg-gradient-to-b from-transparent via-transparent to-black/60 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>

                                    <!-- Category Badge -->
                                    <div class="absolute top-4 right-4 backdrop-blur-md bg-white/10 border border-white/20 rounded-full pl-2 pr-3 py-1 flex items-center gap-2">
                                        <span class="w-2 h-2 rounded-full" style="background-color: <?php echo esc_attr($category_color); ?>"></span>
                                        <span class="text-white text-sm font-medium"><?php echo esc_html($item['category']); ?></span>
                                    </div>

                                    <!-- Title always visible, description on hover -->
                                    <div class="absolute inset-x-0 bottom-0 p-6 translate-y-10 group-hover:translate-y-0 transition-transform duration-500">
                                        <div class="space-y-3 text-white">
                                            <h3 class="text-base font-semibold line-clamp-1"><?php echo esc_html($item['title']); ?></h3>
                                            <p class="text-sm text-white/90 opacity-0 group-hover:opacity-100 transition-opacity duration-500 line-clamp-2"><?php echo esc_html($item['description']); ?></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

            </div>
        </section>
        <!-- Gallery Popup/Lightbox -->
        <div id="galleryPopup" class="fixed inset-0 z-50 flex items-center justify-center backdrop-blur-sm bg-black/60 opacity-0 pointer-events-none transition-opacity duration-300">
            <div class="relative mx-auto">
                <!-- Close Button -->
                <button id="closePopup" class="absolute top-4 right-4 z-10 w-10 h-10 flex items-center justify-center rounded-full bg-gray-800/50 hover:bg-gray-800/70 text-white transition-colors">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
                    </svg>
                </button>

                <!-- Navigation Buttons -->
                <button id="prevImage" class="absolute left-4 top-1/2 -translate-y-1/2 z-10 w-12 h-12 flex items-center justify-center rounded-full bg-gray-800/50 hover:bg-gray-800/70 text-white transition-colors">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7" />
                    </svg>
                </button>

                <button id="nextImage" class="absolute right-4 top-1/2 -translate-y-1/2 z-10 w-12 h-12 flex items-center justify-center rounded-full bg-gray-800/50 hover:bg-gray-800/70 text-white transition-colors">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7" />
                    </svg>
                </button>

                <!-- Image Container (with natural sizing) -->
                <div class="relative rounded-2xl overflow-hidden shadow-2xl">
                    <img id="popupImage" class="max-w-screen-lg max-h-[90vh]" src="" alt="">

                    <!-- Image Info overlay -->
                    <div id="imageInfo" class="absolute bottom-0 left-0 right-0 p-6 bg-gradient-to-t from-black/80 to-transparent">
                        <h3 id="imageTitle" class="text-xl sm:text-2xl font-semibold text-white"></h3>
                        <p id="imageDescription" class="mt-2 text-white/90"></p>
                        <div id="imageCategory" class="mt-3 inline-flex items-center gap-2 backdrop-blur-md bg-white/10 border border-white/20 rounded-full pl-2 pr-3 py-1">
                            <span id="categoryDot" class="w-2 h-2 rounded-full"></span>
                            <span id="categoryName" class="text-white text-sm font-medium"></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    <?php
    }

    /**
     * Render widget output in the editor.
     */
    protected function content_template()
    {
    ?>
        <# var primaryColor=settings.primary_color || '#FFB600' ; #>
            <section class="gallery-section relative py-16 sm:py-20 lg:py-20 overflow-hidden">
                <!-- Background Elements -->
                <div class="absolute inset-0 -z-10">
                    <!-- Gradient Orbs -->
                    <div class="absolute top-0 left-0 w-full h-full overflow-hidden">
                        <div style="background-color: <?php echo esc_attr($primary_color); ?>20" class="bg-fade-effect absolute top-1/4 -left-20 w-96 h-96 rounded-full mix-blend-multiply filter blur-3xl"></div>
                        <div style="background-color: <?php echo esc_attr($primary_color); ?>20" class="absolute bottom-1/4 right-0 w-96 h-96 rounded-full mix-blend-multiply filter blur-3xl animation-delay-2000"></div>
                    </div>
                </div>

                <div class="max-w-7xl 2xl:max-w-8xl mx-auto px-4">
                    <!-- Section Header -->
                    <div class="relative mb-12 sm:mb-16 text-center z-10">
                        <div class="absolute -top-10 left-1/2 -translate-x-1/2 text-[80px] sm:text-[120px] font-bold text-gray-100 select-none opacity-60 leading-none -z-10">
                            {{{ settings.background_text }}}
                        </div>
                        <h2 class="relative text-5xl font-bold text-gray-900">
                            {{{ settings.heading_text }}}
                            <span style="color: {{ primaryColor }}">
                                {{{ settings.heading_span }}}
                            </span>
                        </h2>
                        <p class="mt-4 text-xl text-gray-600 max-w-2xl mx-auto">
                            {{{ settings.section_description }}}
                        </p>
                    </div>

                    <!-- Filter Tabs -->
                    <div class="filter-buttons flex flex-wrap justify-center gap-6 sm:gap-8 mb-12" id="filterButtons">
                        <# _.each(settings.filter_tabs, function(tab) { #>
                            <button style="--active-color: {{ primaryColor }};" data-category="{{ tab.category }}" class="filter-btn px-6 py-3 rounded-xl {{ tab.is_active === 'yes' ? 'active-tab' : '' }} font-medium hover:border-primary-amber   transition-colors">
                                {{{ tab.filter_name }}}
                            </button>
                            <# }); #>
                    </div>

                    <!-- Gallery Grid -->
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 sm:gap-8" id="galleryGrid">
                        <# _.each(settings.gallery_items, function(item) {
                            // Find the matching filter tab to get the category color
                            var categoryColor='#60A5FA' ; // Default color
                            _.each(settings.filter_tabs, function(tab) {
                            if (tab.category===item.category) {
                            categoryColor=tab.category_color;
                            }
                            });
                            #>
                            <div class="group relative" data-category="{{ item.category }}">
                                <div class="relative bg-white/80 backdrop-blur-xl rounded-2xl overflow-hidden shadow-lg hover:shadow-xl transition-all duration-500">
                                    <!-- Image -->
                                    <div class="relative aspect-[4/3] overflow-hidden">
                                        <# if (item.image.url) { #>
                                            <img src="{{ item.image.url }}" alt="{{ item.title }}"
                                                class="w-full h-full object-cover transform group-hover:scale-110 transition-transform duration-700"
                                                fetchpriority="high"
                                                decoding="async" />
                                            <# } #>

                                                <!-- Permanent gradient for title only -->
                                                <div class="absolute inset-x-0 bottom-0 h-40 bg-gradient-to-t from-black/90 to-transparent"></div>

                                                <!-- Hover gradient -->
                                                <div class="absolute inset-0 bg-gradient-to-b from-transparent via-transparent to-black/60 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>

                                                <!-- Category Badge -->
                                                <div class="absolute top-4 right-4 backdrop-blur-md bg-white/10 border border-white/20 rounded-full pl-2 pr-3 py-1 flex items-center gap-2">
                                                    <span class="w-2 h-2 rounded-full" style="background-color: {{ categoryColor }}"></span>
                                                    <span class="text-white text-sm font-medium">{{{ item.category }}}</span>
                                                </div>

                                                <!-- Title always visible, description on hover -->
                                                <div class="absolute inset-x-0 bottom-0 p-6 translate-y-10 group-hover:translate-y-0 transition-transform duration-500">
                                                    <div class="space-y-3 text-white">
                                                        <h3 class="text-base font-semibold line-clamp-1">{{{ item.title }}}</h3>
                                                        <p class="text-sm text-white/90 opacity-0 group-hover:opacity-100 transition-opacity duration-500 line-clamp-2">{{{ item.description }}}</p>
                                                    </div>
                                                </div>
                                    </div>
                                </div>
                            </div>
                            <# }); #>
                    </div>
                </div>
            </section>


    <?php
    }
}
