<?php

/**
 * Project List Section Widget for Elementor
 */

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

if (!defined('ABSPATH')) exit; // Exit if accessed directly
class Projects_List_Widget extends Widget_Base
{

    public function get_name()
    {
        return 'projects_list';
    }

    public function get_title()
    {
        return esc_html__('Projects List', 'dhaara-addon');
    }

    public function get_icon()
    {
        return 'eicon-gallery-grid';
    }

    public function get_categories()
    {
        return ['dhaara'];
    }

    protected function register_controls()
    {

        // Header Section
        $this->start_controls_section(
            'section_header',
            [
                'label' => __('Header', 'dhaara-addon'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'background_text',
            [
                'label' => 'Background Text',
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => 'PROJECTS',
            ]
        );


        $this->add_control(
            'heading_text',
            [
                'label' => esc_html__('Heading', 'dhaara-addon'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Transforming Visions', 'dhaara-addon'),
            ]
        );

        $this->add_control(
            'heading_span',
            [
                'label' => esc_html__('Heading Span Text', 'dhaara-addon'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Into Reality', 'dhaara-addon'),
            ]
        );
        $this->add_control(
            'section_description',
            [
                'label' => __('Section Description', 'dhaara-addon'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('Discover our portfolio of groundbreaking projects that showcase innovative design, sustainable construction, and architectural excellence.', 'dhaara-addon'),
            ]
        );

        $this->end_controls_section();


        // Content Section
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Content Settings', 'dhaara-addon'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label' => esc_html__('Projects Per Page', 'dhaara-addon'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 9,
                'min' => 1,
                'max' => 48,
            ]
        );


        $this->add_control(
            'show_pagination',
            [
                'label' => esc_html__('Show Pagination', 'dhaara-addon'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        // Style Section
        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('Colors', 'dhaara-addon'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'primary_color',
            [
                'label' => esc_html__('Primary Color', 'dhaara-addon'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'global' => [
                    'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Colors::COLOR_PRIMARY,
                ],

            ]
        );
        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $primary_color = !empty($settings['primary_color']) ? $settings['primary_color'] : '#FFB600';
        // Query projects
        $args = [
            'post_type' => 'project',
            'posts_per_page' => $settings['posts_per_page'],
            'paged' => get_query_var('paged') ? get_query_var('paged') : 1,
        ];

        $projects_query = new WP_Query($args);

        // Start output 
?>
        <section class="project-list relative py-16 sm:py-20 lg:py-20 overflow-hidden z-10">
            <!-- Section Header -->
            <div class="relative mb-12 sm:mb-16 text-center z-10">
                <div class="absolute -top-10 left-1/2 -translate-x-1/2 text-[80px] sm:text-[120px] font-bold text-gray-100 select-none opacity-0 md:opacity-60 leading-none -z-10">
                    <?php echo esc_html($settings['background_text']); ?>
                </div>
                <h2 class="relative text-3xl sm:text-4xl lg:text-5xl font-bold text-gray-900 fade-in-effect ">
                    <?php echo esc_html($settings['heading_text']); ?>
                    <span style="color: <?php echo esc_attr($primary_color); ?>"><?php echo esc_html($settings['heading_span']); ?></span>
                </h2>
                <p class="mt-4 text-xl text-gray-600 max-w-2xl mx-auto fade-in-effect effect-1">
                    <?php echo esc_html($settings['section_description']); ?>
                </p>
            </div>


            <!-- Projects Grid -->
            <div class="max-w-7xl 2xl:max-w-8xl mx-auto px-4">
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 sm:gap-8">
                    <?php
                    if ($projects_query->have_posts()):
                        while ($projects_query->have_posts()): $projects_query->the_post();
                            $document = \Elementor\Plugin::$instance->documents->get(get_the_ID());
                            if ($document) {
                                $projectSettings = $document->get_settings_for_display();
                                $progress = isset($projectSettings['project_progress']) ? $projectSettings['project_progress'] : '0';
                                $location = isset($projectSettings['project_location']) ? $projectSettings['project_location'] : '';
                                $completion_date = isset($projectSettings['project_completion_date']) ? $projectSettings['project_completion_date'] : '';
                            }

                            $status = wp_get_post_terms(get_the_ID(), 'project_status', ['fields' => 'names']);
                            $status = !empty($status) ? $status[0] : '';
                    ?>
                            <div class="group">
                                <div class="relative bg-white/80 backdrop-blur-xl rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500 h-full flex flex-col">
                                    <!-- Image Container -->
                                    <div class="relative aspect-[4/3] overflow-hidden">
                                        <?php if (has_post_thumbnail()): ?>
                                            <?php the_post_thumbnail('large', ['class' => 'w-full h-full object-cover transform group-hover:scale-110 transition-transform duration-700']); ?>
                                        <?php endif; ?>
                                        <div class="absolute inset-0 bg-gradient-to-b from-black/5 via-black/5 to-black/60 group-hover:to-black/70 transition-all duration-500"></div>
                                        <!-- Category Badge -->
                                        <?php
                                        $categories = get_the_terms(get_the_ID(), 'project_category');
                                        if ($categories): ?>
                                            <div class="absolute top-4 right-4 backdrop-blur-md bg-white/10 border border-white/20 rounded-full pl-2 pr-3 py-1 flex items-center gap-2">
                                                <span class="text-white text-sm font-medium"><?php echo esc_html($categories[0]->name); ?></span>
                                            </div>
                                        <?php endif; ?>

                                        <!-- Bottom Info Layer -->
                                        <div class="absolute bottom-0 left-0 right-0 p-6 translate-y-6 group-hover:translate-y-0 transition-transform duration-500">
                                            <div class="space-y-3 text-white">
                                                <h3 class="text-lg font-semibold"><?php the_title(); ?></h3>
                                                <p class="text-sm text-white/90 opacity-0 group-hover:opacity-100 transition-opacity duration-500 line-clamp-2">
                                                    <?php echo wp_trim_words(get_the_excerpt(), 20); ?>
                                                </p>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Stats Section -->
                                    <div class="p-6 space-y-4 flex-1 flex flex-col justify-between">
                                        <div class="flex items-center justify-between text-sm">
                                            <?php if (!empty($location)): ?>
                                                <div class="space-y-1">
                                                    <p class="text-gray-500 my-2"><?php echo esc_html__('Location', 'dhaara-addon'); ?></p>
                                                    <p class="font-medium text-gray-900"><?php echo esc_html($location); ?></p>
                                                </div>
                                            <?php endif; ?>

                                            <?php if (!empty($status)): ?>
                                                <div class="space-y-1">
                                                    <p class="text-gray-500 my-2"><?php echo esc_html__('Status', 'dhaara-addon'); ?></p>
                                                    <p class="text-gray-500"></p>
                                                    <p class="font-medium"><?php echo esc_html($status); ?></p>
                                                </div>
                                            <?php endif; ?>
                                        </div>

                                        <!-- Progress Bar -->
                                        <div class="space-y-2">
                                            <div class="flex justify-between text-sm">
                                                <span class="text-gray-500">Project Progress</span>
                                                <span class="text-gray-900 font-medium"><?php echo esc_html($progress); ?>%</span>
                                            </div>
                                            <div class="h-2 bg-gray-100 rounded-full overflow-hidden">
                                                <div class="h-full bg-gradient-to-r bg-primary-amber rounded-full" style="width: <?php echo esc_attr($progress); ?>%"></div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Action Footer -->
                                    <div class="px-6 pb-6">
                                        <a href="<?php echo esc_url(get_the_permalink()); ?>" class="inline-flex items-center font-semibold group-hover:translate-x-2 transition-all duration-300" style="color: <?php echo esc_attr($primary_color); ?>">
                                            <?php echo esc_html__('View Project Details', 'dhaara-addon'); ?>
                                            <svg class="w-5 h-5 ml-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 8l4 4m0 0l-4 4m4-4H3" />
                                            </svg>
                                        </a>
                                    </div>
                                </div>
                            </div>
                    <?php
                        endwhile;
                        wp_reset_postdata();
                    endif;
                    ?>
                </div>

                <?php if ($settings['show_pagination'] === 'yes' && $projects_query->max_num_pages > 1): ?>
                    <!-- Pagination -->
                    <div class="w-full mx-auto mt-8 pb-24 fade-in-effect effect-1">
                        <div class="relative">
                            <div class="absolute inset-0 bg-gradient-to-r from-amber-50/50 to-purple-50/50 backdrop-blur-xl rounded-2xl"></div>
                            <div class="relative bg-white/40 backdrop-blur-xl border border-white/20 rounded-2xl p-6 shadow-xl shadow-black/5">
                                <div class="flex flex-col md:flex-row items-center justify-between gap-6 w-full">
                                    <?php $this->render_pagination($projects_query); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </section>
    <?php
    }

    protected function render_pagination($query)
    {
        if ($query->max_num_pages <= 1) return;
        $settings = $this->get_settings_for_display();
        $current_page = max(1, get_query_var('paged'));
        $total_pages = $query->max_num_pages;

    ?>

        <!-- Results Per Page -->
        <div class="flex items-center gap-3">
            <span class="text-sm text-gray-500"><?php echo esc_html__("Display:", "dhaara-addon"); ?> </span>
            <div class="relative group">
                <?php echo esc_html($settings['posts_per_page']); ?> <?php echo esc_html__('Projects', 'dhaara-addon'); ?>
            </div>
        </div>

        <!-- Pagination Controls -->
        <div class="flex items-center gap-2">
            <!-- Previous Button -->
            <?php if ($current_page > 1) : ?>
                <a href="<?php echo get_pagenum_link($current_page - 1); ?>" class="group relative px-4 py-2.5 rounded-xl bg-white/50 border border-gray-100 font-medium text-gray-600 hover:text-gray-900 hover:border-transparent hover:shadow-lg hover:shadow-amber-500/20 transition-all duration-300">
                    <div class="flex items-center gap-2">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                        </svg>
                        <span><?php echo esc_html__('Previous', 'dhaara-addon'); ?></span>
                    </div>
                </a>
            <?php endif; ?>

            <!-- Page Numbers -->
            <div class="flex items-center gap-2">
                <?php
                $start_number = max(1, min($current_page - 2, $total_pages - 4));
                $end_number = min($total_pages, max(5, $current_page + 2));

                // First page
                if ($start_number > 1) : ?>
                    <a href="<?php echo get_pagenum_link(1); ?>" class="w-10 h-10 rounded-xl bg-white/50 border border-gray-100 font-medium text-gray-600 hover:text-gray-900 hover:border-transparent hover:shadow-lg hover:shadow-amber-500/20 transition-all duration-300 flex items-center justify-center">1</a>
                    <?php if ($start_number > 2) : ?>
                        <span class="w-10 h-10 rounded-xl bg-white/50 border border-gray-100 font-medium text-gray-400 flex items-center justify-center">...</span>
                    <?php endif;
                endif;

                // Page numbers
                for ($i = $start_number; $i <= $end_number; $i++) {
                    $is_current = $i === $current_page;
                    ?>
                    <a href="<?php echo get_pagenum_link($i); ?>"
                        class="w-10 h-10 rounded-xl flex items-center justify-center font-medium <?php echo $is_current ? 'bg-gradient-to-r bg-primary-amber text-white shadow-lg shadow-amber-500/25' : 'bg-white/50 border border-gray-100 text-gray-600 hover:text-gray-900 hover:border-transparent hover:shadow-lg hover:shadow-amber-500/20 transition-all duration-300'; ?>">
                        <?php echo $i; ?>
                    </a>
                    <?php
                }

                // Last page
                if ($end_number < $total_pages) :
                    if ($end_number < $total_pages - 1) : ?>
                        <span class="w-10 h-10 rounded-xl bg-white/50 border border-gray-100 font-medium text-gray-400 flex items-center justify-center">...</span>
                    <?php endif; ?>
                    <a href="<?php echo get_pagenum_link($total_pages); ?>" class="w-10 h-10 rounded-xl bg-white/50 border border-gray-100 font-medium text-gray-600 hover:text-gray-900 hover:border-transparent hover:shadow-lg hover:shadow-amber-500/20 transition-all duration-300 flex items-center justify-center"><?php echo $total_pages; ?></a>
                <?php endif; ?>
            </div>

            <!-- Next Button -->
            <?php if ($current_page < $total_pages) : ?>
                <a href="<?php echo get_pagenum_link($current_page + 1); ?>" class="group relative px-4 py-2.5 rounded-xl bg-white/50 border border-gray-100 font-medium text-gray-600 hover:text-gray-900 hover:border-transparent hover:shadow-lg hover:shadow-amber-500/20 transition-all duration-300">
                    <div class="flex items-center gap-2">
                        <span><?php echo esc_html__('Next', 'dhaara-addon'); ?></span>
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                        </svg>
                    </div>
                </a>
            <?php endif; ?>
        </div>

        <!-- Page Info with Progress Bar -->
        <div class="hidden lg:flex items-center gap-3">
            <div class="h-1.5 w-20 bg-gray-100 rounded-full overflow-hidden">
                <div class="h-full bg-gradient-to-r bg-primary-amber rounded-full" style="width: <?php echo ($current_page / $total_pages) * 100; ?>%"></div>
            </div>
            <span class="text-sm text-gray-500">
                <?php echo esc_html__('Page', 'dhaara-addon'); ?> <span class="font-medium text-gray-900"><?php echo $current_page; ?></span> of <span class="font-medium text-gray-900"><?php echo $total_pages; ?></span>
            </span>
        </div>

<?php
    }
}
